#!/bin/bash
# KAOS — double-click to install (first run) or launch (every run after)

KAOS_DIR="$(cd "$(dirname "$0")" && pwd)"
VENV="$KAOS_DIR/.venv"
MODEL_URL="https://storage.googleapis.com/mediapipe-models/hand_landmarker/hand_landmarker/float16/latest/hand_landmarker.task"
MODEL_FILE="$KAOS_DIR/hand_landmarker.task"
PORT=5050

RED='\033[0;31m'; GREEN='\033[0;32m'; YELLOW='\033[1;33m'
BOLD='\033[1m'; RESET='\033[0m'

ok()   { echo -e "${GREEN}✓${RESET} $*"; }
warn() { echo -e "${YELLOW}⚠${RESET}  $*"; }
err()  { echo -e "${RED}✗  $*${RESET}"; }
step() { echo -e "\n${BOLD}── $* ${RESET}"; }
die()  { err "$*"; echo ""; read -p "Press Enter to close..."; exit 1; }

# ── Stale-PID / already-running check ────────────────────────────────────────
PID_FILE="$KAOS_DIR/.kaos.pid"
if [[ -f "$PID_FILE" ]]; then
    OLD_PID=$(cat "$PID_FILE")
    if kill -0 "$OLD_PID" 2>/dev/null; then
        echo "KAOS is already running (PID $OLD_PID) — stopping it..."
        kill "$OLD_PID"
        sleep 1
    fi
    rm -f "$PID_FILE"
fi

# ── Quick launch (already installed) ─────────────────────────────────────────
if [[ -d "$VENV/bin" ]] && "$VENV/bin/python" -c "import flask, mediapipe, mido" 2>/dev/null; then
    echo -e "${BOLD}Starting KAOS...${RESET}  →  http://127.0.0.1:${PORT}"
    source "$VENV/bin/activate"
    exec python "$KAOS_DIR/main.py"
fi

# ── First run: install ────────────────────────────────────────────────────────
echo -e "${BOLD}"
echo "  ██╗  ██╗ █████╗  ██████╗ ███████╗"
echo "  ██║ ██╔╝██╔══██╗██╔═══██╗██╔════╝"
echo "  █████╔╝ ███████║██║   ██║███████╗"
echo "  ██╔═██╗ ██╔══██║██║   ██║╚════██║"
echo "  ██║  ██╗██║  ██║╚██████╔╝███████║"
echo "  ╚═╝  ╚═╝╚═╝  ╚═╝ ╚═════╝ ╚══════╝"
echo -e "${RESET}"
echo "  First-run setup. This takes about 2 minutes."
echo ""

# 1. macOS version
step "Checking macOS..."
MACOS_MAJOR=$(sw_vers -productVersion | cut -d. -f1)
if (( MACOS_MAJOR < 11 )); then
    die "macOS 11 or later required (you have $(sw_vers -productVersion))"
fi
ok "macOS $(sw_vers -productVersion)"

# 2. Homebrew
step "Checking Homebrew..."
if ! command -v brew &>/dev/null; then
    echo ""
    err "Homebrew is not installed. Install it first, then double-click again:"
    echo ""
    echo '    /bin/bash -c "$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)"'
    echo ""
    read -p "Press Enter to close..."
    exit 1
fi
ok "Homebrew $(brew --version | head -1 | awk '{print $2}')"

# 3. Python 3.11
step "Checking Python 3.11..."
if ! brew list python@3.11 &>/dev/null; then
    echo "  Installing python@3.11 via Homebrew..."
    brew install python@3.11
fi
PYTHON311="$(brew --prefix python@3.11)/bin/python3.11"
[[ -x "$PYTHON311" ]] || die "python3.11 not found at $PYTHON311"
ok "$($PYTHON311 --version)"

# 4. Virtual environment
step "Creating virtual environment..."
if [[ ! -d "$VENV" ]]; then
    "$PYTHON311" -m venv "$VENV"
fi
source "$VENV/bin/activate"
pip install --quiet --upgrade pip
ok ".venv ready"

# 5. Install packages
step "Installing packages (flask, mediapipe, opencv, mido...)..."
pip install --requirement "$KAOS_DIR/requirements.txt"
ok "All packages installed"

# 6. Model file
step "Checking hand landmark model..."
if [[ ! -f "$MODEL_FILE" ]]; then
    echo "  Downloading hand_landmarker.task (~7.5 MB)..."
    curl --location --progress-bar --output "$MODEL_FILE" "$MODEL_URL" \
        || die "Download failed — check your internet connection"
fi
ok "hand_landmarker.task ($(du -sh "$MODEL_FILE" | cut -f1))"

# ── IAC Driver setup ──────────────────────────────────────────────────────────
echo ""
echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"
echo -e "${BOLD}  One-time IAC Driver setup required${RESET}"
echo -e "${BOLD}━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━${RESET}"
echo ""
echo "  Audio MIDI Setup will open in 3 seconds. Then:"
echo ""
echo "    1. Window → Show MIDI Studio"
echo "    2. Double-click  IAC Driver"
echo "    3. Check  'Device is online'"
echo "    4. Confirm Bus 1 exists → Apply → close"
echo ""
echo "  In Logic Pro: set MIDI input to  IAC Driver Bus 1"
echo ""
sleep 3
open -a "Audio MIDI Setup" 2>/dev/null || warn "Open Audio MIDI Setup manually"

echo ""
read -p "  Press Enter when IAC Driver is configured and ready to launch KAOS..."

# ── Launch ────────────────────────────────────────────────────────────────────
echo ""
echo -e "${BOLD}Starting KAOS...${RESET}  →  http://127.0.0.1:${PORT}"
echo ""
exec python "$KAOS_DIR/main.py"
